/// \file DetectorConstruction.cc
/// \brief Implementation of the DetectorConstruction class

#include "DetectorConstruction.hh"

#include "G4Material.hh"
#include "G4NistManager.hh"

#include "G4Box.hh"
#include "G4LogicalVolume.hh"
#include "G4PVPlacement.hh"
#include "G4PVReplica.hh"
#include "G4AutoDelete.hh"

#include "G4GeometryManager.hh"
#include "G4PhysicalVolumeStore.hh"
#include "G4LogicalVolumeStore.hh"
#include "G4SolidStore.hh"

#include "G4VisAttributes.hh"
#include "G4Colour.hh"

#include "G4PhysicalConstants.hh"
#include "G4SystemOfUnits.hh"

G4ThreadLocal

DetectorConstruction::DetectorConstruction()
 : G4VUserDetectorConstruction(),
   fSciLV(0),
   fCheckOverlaps(true)
{
}

DetectorConstruction::~DetectorConstruction()
{ 
}

G4VPhysicalVolume* DetectorConstruction::Construct()
{
  // Define materials 
  DefineMaterials();
  // Define volumes
  return DefineVolumes();
}

void DetectorConstruction::DefineMaterials()
{ 
  // Lead material defined using NIST Manager
  G4NistManager* nistManager = G4NistManager::Instance();
  nistManager->FindOrBuildMaterial("G4_Pb");
  nistManager->FindOrBuildMaterial("G4_PLASTIC_SC_VINYLTOLUENE");
  
  // Liquid argon material
  G4double a;  // mass of a mole;
  G4double z;  // z=mean number of protons;  
  G4double density; 

  // Vacuum
  new G4Material("Galactic", z=1., a=1.01*g/mole,density= universe_mean_density,
                  kStateGas, 2.73*kelvin, 3.e-18*pascal);

  // Print materials
  G4cout << *(G4Material::GetMaterialTable()) << G4endl;
}

G4VPhysicalVolume* DetectorConstruction::DefineVolumes()
{
  // Geometry parameters
  G4int nofLayers = 30; // if changed, energy in layer array has to be adapted in EventAction
  
  G4double sciThickness = 5.*mm;
  G4double absThickness = 0.*mm;
  G4double calorSizeXY  = 50.*cm;

  G4double layerThickness = sciThickness + absThickness;
  G4double calorThickness = nofLayers * layerThickness;
  G4double worldSizeXY = 1.2 * calorSizeXY;
  G4double worldSizeZ  = 1.2 * calorThickness; 
  
  // Get materials
  G4Material* defaultMaterial = G4Material::GetMaterial("Galactic");
  G4Material* absMaterial = G4Material::GetMaterial("G4_Pb");
  G4Material* sciMaterial = G4Material::GetMaterial("G4_PLASTIC_SC_VINYLTOLUENE");
  
  if ( ! defaultMaterial || ! sciMaterial ) {
    G4ExceptionDescription msg;
    msg << "Cannot retrieve materials already defined."; 
    G4Exception("DetectorConstruction::DefineVolumes()",
      "MyCode0001", FatalException, msg);
  }  
   
  //     
  // World
  //
  G4VSolid* worldS 
    = new G4Box("World",           // its name
                 worldSizeXY/2, worldSizeXY/2, worldSizeZ/2); // its size
                         
  G4LogicalVolume* worldLV
    = new G4LogicalVolume(
                 worldS,           // its solid
                 defaultMaterial,  // its material
                 "World");         // its name
                                   
  G4VPhysicalVolume* worldPV
    = new G4PVPlacement(
                 0,                // no rotation
                 G4ThreeVector(),  // at (0,0,0)
                 worldLV,          // its logical volume                         
                 "World",          // its name
                 0,                // its mother  volume
                 false,            // no boolean operation
                 0,                // copy number
                 fCheckOverlaps);  // checking overlaps 
  
  //                               
  // Calorimeter
  //  
  G4VSolid* calorimeterS
    = new G4Box("Calorimeter",     // its name
                 calorSizeXY/2, calorSizeXY/2, calorThickness/2); // its size
                         
  G4LogicalVolume* calorLV
    = new G4LogicalVolume(
                 calorimeterS,     // its solid
                 defaultMaterial,  // its material
                 "Calorimeter");   // its name

  new G4PVPlacement(
                 0,                // no rotation
                 G4ThreeVector(),  // at (0,0,0)
                 calorLV,          // its logical volume                         
                 "Calorimeter",    // its name
                 worldLV,          // its mother  volume
                 false,            // no boolean operation
                 0,                // copy number
                 fCheckOverlaps);  // checking overlaps 
  
  //                               
  // Scintillator
  //
  G4VSolid* sciS 
    = new G4Box("Sci",             // its name
                 calorSizeXY/2, calorSizeXY/2, sciThickness/2); // its size
                         
  fSciLV
    = new G4LogicalVolume(
                 sciS,             // its solid
                 sciMaterial,      // its material
                 "Sci");           // its name
    
  G4VSolid* absS;
  G4LogicalVolume* absLV;
  
  if (absThickness > 0) {
    absS 
      = new G4Box("Abs",             // its name
                   calorSizeXY/2, calorSizeXY/2, absThickness/2); // its size
                           
    absLV
      = new G4LogicalVolume(
                   absS,             // its solid
                   absMaterial,      // its material
                   "Abs");           // its name
  }                    
    
  for (int i = 0; i < nofLayers; i++) {  
    if (absThickness > 0) {
      new G4PVPlacement(
                   0,                // no rotation
                   G4ThreeVector(0., 0., -(calorThickness/2) + (absThickness/2) + i * (layerThickness)), // its position
                   absLV,            // its logical volume                         
                   "Abs",            // its name
                   calorLV,          // its mother  volume
                   false,            // no boolean operation
                   i,                // copy number
                   fCheckOverlaps);  // checking overlaps
    }
    new G4PVPlacement(
                 0,                // no rotation
                 G4ThreeVector(0., 0., -(calorThickness/2) + absThickness + (sciThickness/2) + i * (layerThickness)), // its position
                 fSciLV,            // its logical volume                         
                 "Sci",            // its name
                 calorLV,          // its mother  volume
                 false,            // no boolean operation
                 i,                // copy number
                 fCheckOverlaps);  // checking overlaps
  }  

  //                                        
  // Visualization attributes
  //
  worldLV->SetVisAttributes (G4VisAttributes::Invisible);

  G4VisAttributes* simpleBoxVisAtt= new G4VisAttributes(G4Colour(1.0,1.0,1.0));
  simpleBoxVisAtt->SetVisibility(true);
  calorLV->SetVisAttributes(simpleBoxVisAtt);

  //
  // Always return the physical World
  //
  return worldPV;
}
